﻿USE [VoxcoSystem]
GO

IF EXISTS (SELECT 1 FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[sp_GetAppendedInterviewers]') AND type IN (N'P', N'PC'))
   DROP PROCEDURE [dbo].[sp_GetAppendedInterviewers]
GO

CREATE PROCEDURE [dbo].[sp_GetAppendedInterviewers]
    @projectIds [dbo].[IdList] READONLY,
	@unknownAgent NVARCHAR(19)
AS
BEGIN
    SET NOCOUNT ON;

	DECLARE @moreThanOneProject BIT;
	IF EXISTS (SELECT 1 FROM @projectIds HAVING COUNT(*) > 1)
	BEGIN
		SET @moreThanOneProject = 1;
	END
	ELSE
	BEGIN
		SET @moreThanOneProject = 0;
	END

    CREATE TABLE #Interviewers (
        [ID] NVARCHAR(10),
        [Name] NVARCHAR(MAX),
        [NodeNotEmpty] INT,
        [Status] INT,
        [ItemType] INT,
        [ParentID] INT
    );
    CREATE NONCLUSTERED INDEX [IDX_Interviewers_ID] ON #Interviewers ([ID]);

    WITH [Alias_CTE] ([Alias_Id]) AS (
		SELECT DISTINCT [Alias_Id]
		FROM @projectIds p
		INNER JOIN [dbo].[tblObjects] o WITH(NOLOCK) ON p.[Id] = o.[ParentId]
		INNER JOIN [dbo].[tblAlias] a WITH(NOLOCK) ON o.[k_Id] = a.[k_Id]
	)
	INSERT INTO #Interviewers
	SELECT
		u.[k_Id] AS [ID],
		CASE WHEN (a.[FirstName] IS NULL OR LTRIM(RTRIM(a.[FirstName])) = N'') AND (a.[LastName] IS NULL OR LTRIM(RTRIM(a.[LastName])) = N'')
			THEN N'(' + u.[Name] + N')'
			ELSE N'(' + u.[Name] + N') ' + LTRIM(RTRIM(ISNULL(a.[FirstName], N''))) + N' ' + LTRIM(RTRIM(ISNULL(a.[LastName], N'')))
		END AS [Name],
		0 AS [NodeNotEmpty],
		0 AS [Status],
		0 AS [ItemType],
		0 AS [ParentID]
	FROM (
		-- Users
		SELECT a.[Alias_Id]
		FROM [Alias_CTE] a
		UNION
		-- User groups
		SELECT DISTINCT ua.[Alias_Id]
		FROM [Alias_CTE] a
		INNER JOIN [dbo].[tblObjects] g WITH(NOLOCK) ON a.[Alias_Id] = g.[k_Id] AND g.[Type] = 5
		INNER JOIN [dbo].[tblObjects] ug WITH(NOLOCK) ON g.[k_Id] = ug.[ParentId]
		INNER JOIN [dbo].[tblAlias] ua WITH(NOLOCK) ON ug.[k_Id] = ua.[k_Id]
	) ua
	INNER JOIN [dbo].[tblObjects] u WITH(NOLOCK) ON ua.[Alias_Id] = u.[k_Id] AND u.[Type] = 4
	LEFT JOIN [dbo].[tblAgents] a WITH(NOLOCK) ON u.[k_Id] = a.[k_Id];

	DECLARE @projectCursor CURSOR;
	DECLARE @projectId INT;
	DECLARE @projectDbName NVARCHAR(MAX);
	DECLARE @stmt NVARCHAR(MAX);

	SET @projectCursor = CURSOR FAST_FORWARD FOR
	SELECT [Id]
	FROM @projectIds;

	OPEN @projectCursor;

	FETCH NEXT FROM @projectCursor INTO @projectId;

	WHILE @@FETCH_STATUS = 0
	BEGIN
		EXEC [dbo].[vx_sp_GetProjectDBName] @projectId, @projectDbName OUT;

		IF @moreThanOneProject = 1
		BEGIN
			SET @projectId = 0;
		END

		SET @stmt = N'EXEC ' + @projectDbName + N'.[dbo].[sp_GetAppendedInterviewers] @projectId, @unknownAgent';
		EXEC sp_executesql @stmt, N'@projectId INT, @unknownAgent NVARCHAR(19)', @projectId, @unknownAgent;

		FETCH NEXT FROM @projectCursor INTO @projectId;
	END;

	CLOSE @projectCursor;
	DEALLOCATE @projectCursor;

	INSERT INTO #Interviewers
	SELECT DISTINCT
		CONVERT(NVARCHAR(10), A.[userId]) AS [ID], 
		CASE WHEN LTRIM(RTRIM(ISNULL(TA.[FirstName], ''))) + ' ' + LTRIM(RTRIM(ISNULL(TA.[LastName], ''))) = ' ' 
			THEN CONVERT(NVARCHAR(10), A.[userId]) + N' ' + @unknownAgent
			ELSE LTRIM(RTRIM(ISNULL(TA.[FirstName], ''))) + ' ' + LTRIM(RTRIM(ISNULL(TA.[LastName], '')))
		END AS [Name], 
		0 AS [NodeNotEmpty],
		0 AS [Status],
		0 AS [ItemType],
		CASE WHEN @moreThanOneProject = 1
			THEN 0
			ELSE A.[projectId]
		END AS [ParentID]
	FROM @projectIds pr
	INNER JOIN AgentSession A WITH(NOLOCK) ON pr.[Id] = A.[projectId]
	INNER JOIN TblPages P WITH(NOLOCK) ON P.[k_id] = A.[ProjectId] AND P.[name] = 'InterviewerProjectId'
	LEFT JOIN TblAgents TA WITH(NOLOCK) ON TA.[k_id] = A.[userId]
	WHERE NOT EXISTS (
		SELECT 1
		FROM #Interviewers i WITH(NOLOCK)
		WHERE i.[ID] = CONVERT(NVARCHAR(10), A.[userId])
	);
 
	SELECT [ID], [Name], [NodeNotEmpty], [Status], [ItemType], [ParentID]
	FROM #Interviewers
	ORDER BY [Name];

	DROP TABLE #Interviewers;
END
GO
